"use client"

import type React from "react"

import { useState, useEffect, useRef } from "react"
import { useRouter } from "next/navigation"
import { Card, CardContent } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import {
  ArrowLeft,
  TrendingUp,
  TrendingDown,
  Play,
  Pause,
  Zap,
  DollarSign,
  Clock,
  CheckCircle,
  XCircle,
  Target,
  RefreshCw,
} from "lucide-react"
import Image from "next/image"
import { PageTransition } from "@/components/page-transition"
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog"

interface ChartData {
  candle_id: number
  open_time: number
  open: string
  higher: string
  lower: string
  close: string
  volume: number
}

interface ChartResponse {
  date: string
  status: string
  symbol: string
  data: ChartData[]
}

interface TooltipData {
  x: number
  y: number
  candle: ChartData
  visible: boolean
}

interface AutomationState {
  isActive: boolean
  currentProfit: number
  totalTrades: number
  winTrades: number
  lossTrades: number
  currentBalance: string
  lastTransactionId: number | null
  waitingForResult: boolean
  lastTransactionExpirationTime: number | null // Unix timestamp in milliseconds
}

interface TradeHistory {
  id: number
  timestamp: string
  direction: string
  amount: string
  status:
    | "Analisando..."
    | "Traçando estratégia..."
    | "Executando..."
    | "Aguardando resultado..."
    | "Ganhou"
    | "Perdeu"
    | "Cancelada"
    | "Nova oportunidade encontrada..."
  result?: string
  price: string
  transactionExpirationTime?: number // Add this line
}

interface WinNotification {
  id: number
  amount: string
  visible: boolean
}

export default function AutomationPage() {
  const router = useRouter()
  const [chartData, setChartData] = useState<ChartData[]>([])
  const [isLoading, setIsLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)
  const [tooltip, setTooltip] = useState<TooltipData>({ x: 0, y: 0, candle: {} as ChartData, visible: false })
  const [entryValue, setEntryValue] = useState("5.00")
  const [galeMultiplier, setGaleMultiplier] = useState("1.1")
  const [expiration, setExpiration] = useState("1")
  const [profitTarget, setProfitTarget] = useState("100.00")
  const [stopLoss, setStopLoss] = useState("100.00")
  const eventSourceRef = useRef<EventSource | null>(null)
  const [wsStatus, setWsStatus] = useState<"connecting" | "connected" | "disconnected">("connecting")
  const canvasRef = useRef<HTMLCanvasElement>(null)
  const containerRef = useRef<HTMLDivElement>(null)

  // State variables for parameters, initialized to null
  const [symbol, setSymbol] = useState<string | null>(null)
  const [name, setName] = useState<string | null>(null)
  const [imageUrl, setImageUrl] = useState<string | null>(null)
  const [model, setModel] = useState<string | null>(null)

  // Automation state
  const [automation, setAutomation] = useState<AutomationState>({
    isActive: false,
    currentProfit: 0,
    totalTrades: 0,
    winTrades: 0,
    lossTrades: 0,
    currentBalance: "0,00",
    lastTransactionId: null,
    waitingForResult: false,
    lastTransactionExpirationTime: null,
  })

  // Trade history
  const [tradeHistory, setTradeHistory] = useState<TradeHistory[]>([])

  const [currentGaleLevel, setCurrentGaleLevel] = useState(0) // 0 = no gale, 1+ = gale levels
  const [currentEntryAmount, setCurrentEntryAmount] = useState("5.00") // current value considering gale
  const [isExecutingTrade, setIsExecutingTrade] = useState(false)
  const automationActiveRef = useRef(false)
  const executingTradeRef = useRef(false)

  const [winNotification, setWinNotification] = useState<WinNotification | null>(null)
  const [goalNotification, setGoalNotification] = useState<{ visible: boolean; amount: string } | null>(null)
  const [showInsufficientBalanceModal, setShowInsufficientBalanceModal] = useState(false)
  const [remainingTimes, setRemainingTimes] = useState<Record<number, number>>({}) // New state for countdowns

  // States for the leave confirmation modal
  const [showConfirmLeaveModal, setShowConfirmLeaveModal] = useState(false)
  const [remainingTimeForTrade, setRemainingTimeForTrade] = useState(0)
  const [nextNavigationPath, setNextNavigationPath] = useState<string | null>(null)
  const countdownIntervalRef = useRef<NodeJS.Timeout | null>(null)

  // Cooldown state for refresh button
  const [refreshCooldown, setRefreshCooldown] = useState(0)
  const refreshIntervalRef = useRef<NodeJS.Timeout | null>(null)

  // Function to fetch user balance
  const fetchUserBalance = async () => {
    const storedUser = localStorage.getItem("user")
    const userId = storedUser ? JSON.parse(storedUser).id : null

    if (userId) {
      const lastRefreshTime = localStorage.getItem(`lastRefreshTime_${userId}`)
      const now = Date.now()
      const cooldownDuration = 30 * 1000 // 30 seconds

      if (lastRefreshTime && now - Number(lastRefreshTime) < cooldownDuration) {
        const remaining = Math.ceil((cooldownDuration - (now - Number(lastRefreshTime))) / 1000)
        setRefreshCooldown(remaining)
        console.log(`Refresh cooldown active for user ${userId}: ${remaining} seconds remaining.`)
        return // Do not proceed with fetch if cooldown is active
      }
    }

    try {
      const token = localStorage.getItem("token")
      if (!token) {
        console.warn("Token not found, cannot fetch balance.")
        return
      }
      const response = await fetch("/api/user/balance", {
        headers: {
          Authorization: `Bearer ${token}`,
          "Content-Type": "application/json",
        },
      })
      if (response.ok) {
        const data = await response.json()
        setAutomation((prev) => ({
          ...prev,
          currentBalance: data.credit, // Assuming 'credit' is the balance field
        }))
        // Also update localStorage with the latest balance
        if (storedUser) {
          try {
            const parsedUser = JSON.parse(storedUser)
            localStorage.setItem("user", JSON.stringify({ ...parsedUser, credit: data.credit }))
          } catch (e) {
            console.error("Error parsing stored user for balance update on fetch:", e)
          }
        }

        // Set cooldown after successful refresh
        if (userId) {
          localStorage.setItem(`lastRefreshTime_${userId}`, Date.now().toString())
          setRefreshCooldown(30) // Start 30-second cooldown
          if (refreshIntervalRef.current) {
            clearInterval(refreshIntervalRef.current)
          }
          refreshIntervalRef.current = setInterval(() => {
            setRefreshCooldown((prev) => {
              if (prev <= 1) {
                if (refreshIntervalRef.current) {
                  clearInterval(refreshIntervalRef.current)
                }
                return 0
              }
              return prev - 1
            })
          }, 1000)
        }
      } else {
        console.error("Failed to fetch balance:", response.status, await response.text())
      }
    } catch (error) {
      console.error("Error fetching balance:", error)
    }
  }

  // Function to fetch initial chart data
  const fetchInitialChartData = async (currentSymbol: string) => {
    try {
      console.log("🔄 Carregando dados históricos...")
      const response = await fetch(`/api/chart/${currentSymbol}?interval=1&countback=30`)

      if (!response.ok) {
        throw new Error(`Failed to fetch chart data: ${response.status}`)
      }

      const data: ChartResponse = await response.json()

      if (data.status === "success" && data.data) {
        setChartData(data.data)
        console.log(`✅ ${data.data.length} candles históricos carregados`)
        setError(null)
      } else {
        throw new Error("Invalid response format")
      }
    } catch (error) {
      console.error("Error fetching initial chart data:", error)
      setError("Erro ao carregar dados históricos do gráfico")
    } finally {
      setIsLoading(false)
    }
  }

  // Function to connect WebSocket
  const connectWebSocket = (currentSymbol: string) => {
    if (eventSourceRef.current) {
      eventSourceRef.current.close()
    }

    console.log(`🔴 Conectando WebSocket para ${currentSymbol}...`)
    setWsStatus("connecting")

    const newEventSource = new EventSource(`/api/stream/${currentSymbol}`)

    newEventSource.onopen = () => {
      console.log(`✅ WebSocket conectado para ${currentSymbol}`)
      setWsStatus("connected")
    }

    newEventSource.onmessage = (event) => {
      try {
        const newCandle = JSON.parse(event.data)
        console.log("📊 Novo candle recebido:", newCandle)

        setChartData((prevData) => {
          if (prevData.length === 0) return prevData

          const updatedData = [...prevData]
          const lastCandle = updatedData[updatedData.length - 1]

          // If it's the same timestamp, update the current candle
          if (lastCandle && lastCandle.open_time === newCandle.time) {
            updatedData[updatedData.length - 1] = {
              ...lastCandle,
              higher: newCandle.high.toString(),
              lower: newCandle.low.toString(),
              close: newCandle.close.toString(),
              volume: lastCandle.volume,
            }
          } else {
            // New candle - add and keep only the last 30
            const newCandleFormatted = {
              candle_id: Date.now(),
              open: newCandle.open.toString(),
              higher: newCandle.high.toString(),
              lower: newCandle.low.toString(),
              close: newCandle.close.toString(),
              open_time: newCandle.time,
              volume: 0,
            }

            updatedData.push(newCandleFormatted)

            // Keep only the last 30 candles
            if (updatedData.length > 30) {
              return updatedData.slice(-30)
            }
          }

          return updatedData
        })
      } catch (err) {
        console.error("❌ Erro ao processar dados do WebSocket:", err)
      }
    }

    newEventSource.onerror = (error) => {
      console.error("❌ Erro no WebSocket:", error)
      setWsStatus("disconnected")

      // Try to reconnect after 3 seconds
      setTimeout(() => {
        console.log("🔄 Tentando reconectar WebSocket...")
        connectWebSocket(currentSymbol)
      }, 3000)
    }

    eventSourceRef.current = newEventSource
  }

  // Main useEffect to load parameters and initiate data fetching/websocket
  useEffect(() => {
    // Retrieve parameters from localStorage
    const storedParams = localStorage.getItem("aiOperationParams")
    if (storedParams) {
      const params = JSON.parse(storedParams)
      setSymbol(params.symbol)
      setName(params.name)
      setImageUrl(params.img)
      setModel(params.model)

      // Add this new block:
      const storedUser = localStorage.getItem("user")
      if (storedUser) {
        try {
          const parsedUser = JSON.parse(storedUser)
          setAutomation((prev) => ({
            ...prev,
            currentBalance: parsedUser.credit || "0,00",
          }))
        } catch (e) {
          console.error("Error parsing stored user for initial balance:", e)
        }
      }

      // Now that symbol is set, fetch chart data and connect WebSocket
      const currentSymbol = params.symbol
      if (currentSymbol) {
        fetchInitialChartData(currentSymbol)
        connectWebSocket(currentSymbol)
      } else {
        console.warn("AutomationPage: Symbol is null after loading from localStorage. Redirecting.")
        router.push("/ai-operation")
      }
    } else {
      // If no parameters are found in localStorage, redirect back
      router.push("/ai-operation")
    }

    // Fetch initial user balance and set up cooldown
    fetchUserBalance() // Initial fetch
    const storedUser = localStorage.getItem("user")
    const userId = storedUser ? JSON.parse(storedUser).id : null

    if (userId) {
      const lastRefreshTime = localStorage.getItem(`lastRefreshTime_${userId}`)
      const now = Date.now()
      const cooldownDuration = 30 * 1000 // 30 seconds

      if (lastRefreshTime) {
        const elapsed = now - Number(lastRefreshTime)
        if (elapsed < cooldownDuration) {
          setRefreshCooldown(Math.ceil((cooldownDuration - elapsed) / 1000))
          if (refreshIntervalRef.current) {
            clearInterval(refreshIntervalRef.current)
          }
          refreshIntervalRef.current = setInterval(() => {
            setRefreshCooldown((prev) => {
              if (prev <= 1) {
                if (refreshIntervalRef.current) {
                  clearInterval(refreshIntervalRef.current)
                }
                return 0
              }
              return prev - 1
            })
          }, 1000)
        }
      }
    }

    // Cleanup on component unmount
    return () => {
      if (refreshIntervalRef.current) {
        clearInterval(refreshIntervalRef.current)
      }
      if (eventSourceRef.current) {
        console.log("🔴 Fechando WebSocket via ref (cleanup)...")
        eventSourceRef.current.close()
        eventSourceRef.current = null
      }
    }
  }, [router]) // Dependency array includes router for initial load and potential router changes

  useEffect(() => {
    drawChart()
  }, [chartData])

  useEffect(() => {
    const handleResize = () => {
      drawChart()
    }

    window.addEventListener("resize", handleResize)
    return () => window.removeEventListener("resize", handleResize)
  }, [chartData])

  useEffect(() => {
    const interval = setInterval(() => {
      setRemainingTimes((prevTimes) => {
        const newTimes: Record<number, number> = {}
        tradeHistory.forEach((trade) => {
          if (
            (trade.status === "Executando..." || trade.status === "Aguardando resultado...") &&
            trade.transactionExpirationTime
          ) {
            const remaining = trade.transactionExpirationTime - Date.now()
            newTimes[trade.id] = Math.max(0, Math.ceil(remaining / 1000))
          }
        })
        return newTimes
      })
    }, 1000)

    return () => clearInterval(interval)
  }, [tradeHistory]) // Re-run when tradeHistory changes to pick up new trades

  // Effect for countdown
  useEffect(() => {
    // This useEffect is specifically for the leave confirmation modal's countdown.
    // The general trade history countdown is handled by the new useEffect below.
    if (showConfirmLeaveModal && automation.lastTransactionExpirationTime) {
      if (countdownIntervalRef.current) {
        clearInterval(countdownIntervalRef.current)
      }

      const updateCountdown = () => {
        const now = Date.now()
        const remaining = automation.lastTransactionExpirationTime! - now
        if (remaining <= 0) {
          setRemainingTimeForTrade(0)
          if (countdownIntervalRef.current) {
            clearInterval(countdownIntervalRef.current)
            countdownIntervalRef.current = null
          }
        } else {
          setRemainingTimeForTrade(Math.ceil(remaining / 1000))
        }
      }

      updateCountdown() // Initial call
      countdownIntervalRef.current = setInterval(updateCountdown, 1000)
    } else {
      if (countdownIntervalRef.current) {
        clearInterval(countdownIntervalRef.current)
        countdownIntervalRef.current = null
      }
    }

    return () => {
      if (countdownIntervalRef.current) {
        clearInterval(countdownIntervalRef.current)
      }
    }
  }, [showConfirmLeaveModal, automation.lastTransactionExpirationTime])

  // Effect for browser-level navigation (refresh, close tab)
  useEffect(() => {
    const handleBeforeUnload = (event: BeforeUnloadEvent) => {
      if (automation.isActive || automation.waitingForResult) {
        event.preventDefault()
        event.returnValue = "" // Required for Chrome to show the prompt
        return "" // Required for other browsers
      }
    }

    window.addEventListener("beforeunload", handleBeforeUnload)

    return () => {
      window.removeEventListener("beforeunload", handleBeforeUnload)
    }
  }, [automation.isActive, automation.waitingForResult])

  const formatTime = (timestamp: number) => {
    const date = new Date(timestamp * 1000)
    return date.toLocaleTimeString("pt-BR", {
      hour: "2-digit",
      minute: "2-digit",
      hour12: false,
    })
  }

  const getCurrentPriceValue = () => {
    if (chartData.length === 0) return "0"
    return chartData[chartData.length - 1].close
  }

  const getPriceChangeValue = () => {
    if (chartData.length < 2) return { change: 0, percentage: 0 }

    const current = Number.parseFloat(chartData[chartData.length - 1].close)
    const previous = Number.parseFloat(chartData[chartData.length - 2].close)
    const change = current - previous
    const percentage = (change / previous) * 100

    return { change, percentage }
  }

  const getCurrentOHLCValue = () => {
    if (chartData.length === 0) return { open: "0", high: "0", low: "0", close: "0" }

    const current = chartData[chartData.length - 1]
    return {
      open: current.open,
      high: current.higher,
      low: current.lower,
      close: current.close,
    }
  }

  const { change, percentage } = getPriceChangeValue()
  const isPositive = change >= 0
  const ohlc = getCurrentOHLCValue()

  // Status do WebSocket para exibição
  const getWSStatusColor = () => {
    switch (wsStatus) {
      case "connected":
        return "bg-green-500"
      case "connecting":
        return "bg-yellow-500"
      case "disconnected":
        return "bg-red-500"
      default:
        return "bg-gray-500"
    }
  }

  const getWSStatusText = () => {
    switch (wsStatus) {
      case "connected":
        return "LIVE DATA"
      case "connecting":
        return "CONECTANDO"
      case "disconnected":
        return "DESCONECTADO"
      default:
        return "AGUARDANDO"
    }
  }

  const getStatusIcon = (status: TradeHistory["status"]) => {
    if (status === "Ganhou") {
      return <CheckCircle className="w-4 h-4 text-green-400" />
    } else if (status === "Perdeu") {
      return <XCircle className="w-4 h-4 text-red-400" />
    } else {
      return <Clock className="w-4 h-4 text-yellow-400 animate-spin" />
    }
  }

  const getStatusColor = (status: TradeHistory["status"]) => {
    if (status === "Ganhou") return "text-green-400"
    if (status === "Perdeu") return "text-red-400"
    if (status === "Cancelada") return "text-orange-400"
    return "text-yellow-400"
  }

  // Function to handle navigation attempts
  const handleNavigationAttempt = (path: string) => {
    if (automation.isActive || automation.waitingForResult) {
      setNextNavigationPath(path)
      setShowConfirmLeaveModal(true)
    } else {
      router.push(path)
    }
  }

  const exportChartAsBase64 = async (): Promise<string | null> => {
    return new Promise((resolve, reject) => {
      const canvas = canvasRef.current
      if (!canvas) {
        reject(new Error("Canvas reference is not available"))
        return
      }

      try {
        const imgData = canvas.toDataURL("image/png")
        resolve(imgData)
      } catch (error) {
        reject(new Error("Failed to export canvas as base64"))
      }
    })
  }

  const getAIAnalysis = async (): Promise<{ recomendacao: string }> => {
    try {
      const chartBase64 = await exportChartAsBase64()
      if (!chartBase64) {
        throw new Error("Failed to export chart")
      }

      const response = await fetch("/api/ai-analysis", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          imageBase64: chartBase64,
          mode: "automation", // Automation mode - recommendation only
        }),
      })

      if (!response.ok) {
        throw new Error("Failed to get AI analysis")
      }

      return await response.json()
    } catch (error) {
      console.error("Error getting AI analysis:", error)
      throw error
    }
  }

  const calculateGaleAmount = (baseAmount: string, galeMultiplier: string, galeLevel: number) => {
    const base = Number.parseFloat(baseAmount)

    if (galeLevel === 0) {
      return base // No gale, normal value
    }

    // For any gale level > 0, apply the multiplier
    if (galeMultiplier === "1.1") {
      return base * Math.pow(2, galeLevel) // Doubles each level (5 -> 10 -> 20 -> 40...)
    } else if (galeMultiplier === "1.2") {
      return base * Math.pow(3, galeLevel) // Triples each level (5 -> 15 -> 45...)
    } else if (galeMultiplier === "1.3") {
      return base * Math.pow(4, galeLevel) // Quadruples each level (5 -> 20 -> 80...)
    } else if (galeMultiplier === "1.4") {
      return base * Math.pow(5, galeLevel) // Quintuples each level (5 -> 25 -> 125...)
    }

    return base
  }

  const createTransaction = async (direction: string, currentPrice: string, amount?: string) => {
    try {
      const token = localStorage.getItem("token")
      if (!token) {
        throw new Error("Token not found")
      }

      const transactionAmount = amount || entryValue // Use the passed amount or default value

      const response = await fetch("/api/transaction", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          Authorization: `Bearer ${token}`,
        },
        body: JSON.stringify({
          expiration,
          amount: transactionAmount,
          direction: direction === "COMPRA" ? "call" : "put",
          symbol,
          symbolPrice: currentPrice,
          token,
        }),
      })

      if (!response.ok) {
        throw new Error("Failed to create transaction")
      }

      const data = await response.json()
      console.log("=== TRANSAÇÃO CRIADA ===", data)

      setAutomation((prev) => ({
        ...prev,
        lastTransactionId: data.transaction_id,
        waitingForResult: true,
        currentBalance: data.user_credit,
        lastTransactionExpirationTime: new Date(data.expiration_datetime).getTime(), // Store as timestamp
      }))

      return data
    } catch (error) {
      console.error("Error creating transaction:", error)
      throw error
    }
  }

  const checkSettlement = async () => {
    try {
      const token = localStorage.getItem("token")
      if (!token) {
        throw new Error("Token not found")
      }

      const response = await fetch("/api/settlement", {
        headers: { Authorization: `Bearer ${token}` },
      })

      if (!response.ok) {
        throw new Error(`Settlement failed: ${response.status}`)
      }

      return await response.json()
    } catch (error) {
      console.error("Error checking settlement:", error)
      throw error
    }
  }

  const checkTransactionResult = async (transactionId: number) => {
    try {
      const token = localStorage.getItem("token")
      if (!token) {
        throw new Error("Token not found")
      }

      const response = await fetch(`/api/transaction/${transactionId}`, {
        method: "GET",
        headers: {
          Authorization: `Bearer ${token}`,
        },
      })

      if (!response.ok) {
        throw new Error("Failed to check transaction result")
      }

      return await response.json()
    } catch (error) {
      console.error("Error checking transaction result:", error)
      throw error
    }
  }

  const addTradeToHistory = (trade: Omit<TradeHistory, "id">) => {
    const newTrade: TradeHistory = {
      ...trade,
      id: Date.now(),
    }
    setTradeHistory((prev) => [newTrade, ...prev])
    return newTrade.id
  }

  const updateTradeInHistory = (id: number, updates: Partial<TradeHistory>) => {
    setTradeHistory((prev) => prev.map((trade) => (trade.id === id ? { ...trade, ...updates } : trade)))
  }

  // Add function to play win sound
  const playWinSound = () => {
    try {
      const audio = new Audio("/ganhou.mp3")
      audio.volume = 0.7
      audio.play().catch((error) => {
        console.log("Erro ao tocar som de vitória:", error)
      })
    } catch (error) {
      console.log("Erro ao carregar som de vitória:", error)
    }
  }

  const playGoalSound = () => {
    try {
      const audio = new Audio("/meta.mp3")
      audio.volume = 0.7
      audio.play().catch((error) => {
        console.log("Erro ao tocar som de meta:", error)
      })
    } catch (error) {
      console.log("Erro ao carregar som de meta:", error)
    }
  }

  // Add function to show win notification
  const showWinNotification = (winAmount: string) => {
    const notification: WinNotification = {
      id: Date.now(),
      amount: winAmount,
      visible: true,
    }

    setWinNotification(notification)

    // Remove notification after 4 seconds
    setTimeout(() => {
      setWinNotification((prev) => (prev ? { ...prev, visible: false } : null))

      // Clear completely after animation
      setTimeout(() => {
        setWinNotification(null)
      }, 300)
    }, 4000)
  }

  const showGoalNotification = (profitAmount: string) => {
    const notification = {
      visible: true,
      amount: profitAmount,
    }

    setGoalNotification(notification)
    playGoalSound() // Play goal sound
  }

  const dismissGoalNotification = () => {
    setGoalNotification((prev) => (prev ? { ...prev, visible: false } : null))
    // Clear completely after animation
    setTimeout(() => {
      setGoalNotification(null)
    }, 300) // Match the transition duration
  }

  const executeAutomatedTrade = async (forceActive = false, galeLevel = currentGaleLevel, baseAmount = entryValue) => {
    // Use BOTH: the ref and the state for double checking
    if (executingTradeRef.current || isExecutingTrade) {
      console.log("⚠️ BLOQUEADO: Trade já em execução, ignorando chamada duplicada")
      return
    }

    // Check if automation is active
    if (!forceActive && !automationActiveRef.current) {
      console.log("⚠️ Automação pausada, cancelando execução")
      return
    }

    // --- NEW LOGIC FOR INSUFFICIENT BALANCE CHECK ---
    const currentAmount = calculateGaleAmount(entryValue, galeMultiplier, galeLevel) // Value of the next entry
    const currentBalanceNum = Number.parseFloat(automation.currentBalance.replace(/\./g, "").replace(",", ".")) // Convert balance to number

    if (currentBalanceNum < currentAmount) {
      console.log(
        `🛑 Saldo insuficiente para a próxima operação. Saldo: R$ ${currentBalanceNum.toFixed(2)}, Necessário: R$ ${currentAmount.toFixed(2)}. Parando automação.`,
      )
      automationActiveRef.current = false // Stop automation loop
      setAutomation((prev) => ({ ...prev, isActive: false })) // Update state to inactive
      executingTradeRef.current = false
      setIsExecutingTrade(false)
      setShowInsufficientBalanceModal(true) // Show insufficient balance pop-up
      return
    }
    // --- END OF NEW LOGIC FOR INSUFFICIENT BALANCE CHECK ---

    // Set BOTH: ref and state to block other calls
    console.log("🔒 Definindo flags de execução para bloquear chamadas duplicadas")
    executingTradeRef.current = true
    setIsExecutingTrade(true)

    // --- NEW LOGIC FOR STOP LOSS CHECK BEFORE OPERATION ---
    const currentProfit = automation.currentProfit // Get current profit from state
    const stopLossValue = Number.parseFloat(stopLoss) // Convert stopLoss to number
    // currentAmount already calculated above

    const potentialProfitIfLoss = currentProfit - currentAmount // Calculate potential profit if this operation loses

    if (potentialProfitIfLoss <= -stopLossValue) {
      console.log(
        `🛑 Stop loss de R$ ${stopLossValue.toFixed(2)} seria atingido com a próxima operação (R$ ${currentAmount.toFixed(2)}). Lucro atual: R$ ${currentProfit.toFixed(2)}. Potencial: R$ ${potentialProfitIfLoss.toFixed(2)}. Parando automação.`,
      )
      automationActiveRef.current = false // Stop automation loop
      setAutomation((prev) => ({ ...prev, isActive: false })) // Update state to inactive

      // Add a special entry to history indicating Stop Loss was reached
      addTradeToHistory({
        timestamp: new Date().toLocaleTimeString("pt-BR"),
        direction: "STOP LOSS",
        amount: currentAmount.toFixed(2),
        status: "Cancelada", // Or a new status like "Stop Loss Reached"
        result: `Stop Loss Atingido: -R$ ${stopLossValue.toFixed(2)}`,
        price: getCurrentPriceValue(),
      })

      // Exit function, do not proceed with transaction
      executingTradeRef.current = false
      setIsExecutingTrade(false)
      return
    }
    // --- END OF NEW LOGIC FOR STOP LOSS CHECK ---

    console.log("🤖 Executando trade automatizado...")

    try {
      // 1. CALCULATE GALE VALUE FIRST
      // currentAmount already calculated above
      console.log(`💰 DETALHES DO GALE ANTES DA OPERAÇÃO:`)
      console.log(`   - Gale Level atual: ${galeLevel}`)
      console.log(`   - Valor base: R$ ${baseAmount}`)
      console.log(`   - Multiplicador: ${galeMultiplier}`)
      console.log(`   - Valor calculado para esta operação: R$ ${currentAmount}`)

      // 2. Add trade to history with the CORRECT Gale value
      const tradeId = addTradeToHistory({
        timestamp: new Date().toLocaleTimeString("pt-BR"),
        direction: "",
        amount: currentAmount.toString(), // Use the calculated Gale value
        status: "Analisando...",
        price: getCurrentPriceValue(),
        // Add transactionExpirationTime here, it will be updated later when transaction is created
      })

      // Simulate status progress
      setTimeout(() => {
        updateTradeInHistory(tradeId, { status: "Traçando estratégia..." })
      }, 1000)

      setTimeout(() => {
        updateTradeInHistory(tradeId, { status: "Executando..." })
      }, 2000)

      // 3. Get AI analysis
      const analysis = await getAIAnalysis()
      console.log("📊 Análise da IA:", analysis)

      // Update direction in history
      updateTradeInHistory(tradeId, {
        direction: analysis.recomendacao,
        status: "Aguardando resultado...",
      })

      // 4. Get current price
      const currentPrice = getCurrentPriceValue()

      // 5. Create transaction with already calculated Gale value
      try {
        const transaction = await createTransaction(analysis.recomendacao, currentPrice, currentAmount.toString())
        console.log("💰 Transação criada com valor:", currentAmount)

        // Update the trade history entry with the actual expiration time
        updateTradeInHistory(tradeId, {
          transactionExpirationTime: new Date(transaction.expiration_datetime).getTime(),
        })

        setAutomation((prev) => ({
          ...prev,
          lastTransactionId: transaction.transaction_id, // Use transaction.transaction_id
          waitingForResult: true,
          currentBalance: transaction.user_credit, // Use transaction.user_credit
          lastTransactionExpirationTime: new Date(transaction.expiration_datetime).getTime(), // Store as timestamp
        }))

        // Continue with the rest of the transaction logic...
        // 4. Wait for expiration + 2 seconds
        const expirationTime = new Date(transaction.expiration_datetime).getTime()
        const waitTime = expirationTime - Date.now() + 2000

        console.log(`⏰ Aguardando ${waitTime}ms para verificar resultado...`)

        // ... rest of the existing setTimeout logic remains the same ...

        setTimeout(
          async () => {
            try {
              // Check settlement
              const settlement = await checkSettlement()
              console.log("📈 Settlement:", settlement)

              // Update balance
              setAutomation((prev) => {
                // If the result starts with '+', it's positive profit
                // If the result starts with '-', it's a loss
                const isPositiveResult = settlement.amount_result.startsWith("+")
                const resultValue = Number.parseFloat(settlement.amount_result.replace(/[^\d.-]/g, ""))

                // If positive result, add the real value; if negative, subtract only the loss value
                const profitChange = isPositiveResult ? resultValue : -Number.parseFloat(entryValue)

                return {
                  ...prev,
                  currentBalance: settlement.user_credit,
                }
              })

              // 6. Wait 1 second and check transaction result
              setTimeout(async () => {
                try {
                  const result = await checkTransactionResult(transaction.transaction_id)
                  console.log("🎯 Resultado da transação:", result)

                  // In the part where it checks if it won (inside the setTimeout that checks transaction result), replace:
                  const won = result.transaction.status === "Ganhou"
                  const finalStatus = won ? "Ganhou" : "Perdeu"

                  // If won, play sound and show notification
                  if (won) {
                    playWinSound()

                    // Calculate won amount (returns - invested amount)
                    const returnsCents = result.transaction.returns_cents || 0
                    const amountCents = Number.parseFloat(currentAmount.toString()) * 100
                    const winAmount = ((returnsCents - amountCents) / 100).toFixed(2)

                    showWinNotification(`+R$ ${winAmount}`)
                  }

                  // Update history with final result
                  updateTradeInHistory(tradeId, {
                    status: finalStatus,
                    result: won
                      ? `+R$ ${((result.transaction.returns_cents - result.transaction.amount_cents) / 100).toFixed(2)}`
                      : settlement.amount_result,
                  })

                  // GALE LOGIC - APPLY FOR THE NEXT OPERATION
                  let nextGaleLevel = galeLevel
                  let nextEntryAmount = currentEntryAmount

                  if (won) {
                    // If won, reset Gale to base value
                    console.log("✅ Ganhou! Resetando Gale para valor base")
                    nextGaleLevel = 0
                    nextEntryAmount = entryValue
                    setCurrentGaleLevel(0)
                    setCurrentEntryAmount(entryValue)
                  } else {
                    // If lost, increase Gale level for the NEXT operation
                    nextGaleLevel = galeLevel + 1
                    const nextAmount = calculateGaleAmount(baseAmount, galeMultiplier, nextGaleLevel)
                    nextEntryAmount = nextAmount.toString()

                    console.log(`❌ Perdeu! Aumentando Gale de nível ${galeLevel} para nível ${nextGaleLevel}`)
                    console.log(
                      `🎯 Próxima entrada será: R$ ${nextAmount} (base: ${baseAmount}, multiplicador: ${galeMultiplier})`,
                    )

                    setCurrentGaleLevel(nextGaleLevel)
                    setCurrentEntryAmount(nextEntryAmount)
                  }

                  // Store values before updating state
                  let shouldContinue = false
                  let shouldStop = false

                  setAutomation((prev) => {
                    const newTotalTrades = prev.totalTrades + 1
                    const newWinTrades = won ? prev.winTrades + 1 : prev.winTrades
                    const newLossTrades = won ? prev.lossTrades : prev.lossTrades + 1
                    const newCurrentProfit = won
                      ? prev.currentProfit + (result.transaction.returns_cents - result.transaction.amount_cents) / 100
                      : prev.currentProfit - result.transaction.amount_cents / 100

                    // Update localStorage with the latest balance
                    const newBalance = result.transaction.user.balance
                    const storedUser = localStorage.getItem("user")
                    if (storedUser) {
                      try {
                        const parsedUser = JSON.parse(storedUser)
                        localStorage.setItem("user", JSON.stringify({ ...parsedUser, credit: newBalance }))
                      } catch (e) {
                        console.error("Error parsing stored user for balance update after trade:", e)
                      }
                    }

                    // Check if it should continue
                    const targetProfit = Number.parseFloat(profitTarget)
                    const stopLossValue = Number.parseFloat(stopLoss)

                    console.log(
                      `💰 Lucro atual: ${newCurrentProfit}, Meta: ${targetProfit}, Stop Loss: ${-stopLossValue}`,
                    )

                    // Check stop conditions
                    shouldStop = newCurrentProfit >= targetProfit || newCurrentProfit <= -stopLossValue

                    if (newCurrentProfit >= targetProfit) {
                      console.log("🎉 Meta de lucro atingida!")
                      showGoalNotification(newCurrentProfit.toFixed(2)) // Show goal notification
                      automationActiveRef.current = false
                    } else if (newCurrentProfit <= -stopLossValue) {
                      console.log("🛑 Stop loss atingido!")
                      automationActiveRef.current = false
                    } else {
                      // If stop conditions not met, should continue
                      shouldContinue = true
                      automationActiveRef.current = true
                      console.log(`🔄 Automação deve continuar: ${shouldContinue ? "SIM" : "NÃO"}`)
                    }

                    return {
                      ...prev,
                      totalTrades: newTotalTrades,
                      winTrades: newWinTrades,
                      lossTrades: newLossTrades,
                      currentProfit: newCurrentProfit,
                      waitingForResult: false,
                      currentBalance: result.transaction.user.balance,
                      isActive: shouldStop ? false : prev.isActive,
                      lastTransactionExpirationTime: null, // Clear expiration time after result
                    }
                  })

                  // IMPORTANT: Clear execution flags BEFORE scheduling the next execution
                  executingTradeRef.current = false
                  setIsExecutingTrade(false)

                  // Explicit check for debugging
                  console.log(
                    `🔍 Estado da automação: automationActiveRef=${automationActiveRef.current}, shouldStop=${shouldStop}, shouldContinue=${shouldContinue}`,
                  )
                  console.log(`🎲 Próximo Gale Level: ${nextGaleLevel}, Próximo valor: R$ ${nextEntryAmount}`)

                  // Force continuation if stop conditions not met
                  const shouldForceContinue = !shouldStop && automationActiveRef.current

                  // Continue only if the flag is set and there are no stop conditions
                  if (shouldForceContinue) {
                    console.log("🔄 Continuando automação em 5 segundos...")
                    // Use setTimeout to give a longer interval between executions
                    setTimeout(() => {
                      if (automationActiveRef.current) {
                        console.log(`✅ Iniciando próximo trade automatizado com Gale Level ${nextGaleLevel}`)
                        // Pass the correct Gale values for the next execution
                        executeAutomatedTrade(true, nextGaleLevel, baseAmount)
                      } else {
                        console.log("❌ Automação foi pausada durante o intervalo, cancelando próximo trade")
                      }
                    }, 5000)
                  } else {
                    console.log("⏹️ Automação finalizada ou pausada, não agendando próximo trade")
                  }
                } catch (error) {
                  console.error("Erro ao verificar resultado:", error)
                  updateTradeInHistory(tradeId, { status: "Perdeu", result: "Erro" })
                  setAutomation((prev) => ({ ...prev, waitingForResult: false, lastTransactionExpirationTime: null }))
                  // Clear both execution flags
                  executingTradeRef.current = false
                  setIsExecutingTrade(false)
                }
              }, 1000)
            } catch (error) {
              console.error("Erro ao verificar settlement:", error)
              updateTradeInHistory(tradeId, { status: "Perdeu", result: "Erro" })
              setAutomation((prev) => ({ ...prev, waitingForResult: false, lastTransactionExpirationTime: null }))
              // Clear both execution flags
              console.log("🔓 Liberando flags de execução após erro")
              executingTradeRef.current = false
              setIsExecutingTrade(false)

              // Try to reconnect after error if automation is still active
              if (automationActiveRef.current) {
                console.log("🔄 Continuando automação após erro em 10 segundos...")
                setTimeout(() => {
                  if (automationActiveRef.current) {
                    console.log("✅ Reiniciando automação após erro")
                    executeAutomatedTrade()
                  }
                }, 10000)
              }
            }
          },
          Math.max(waitTime, 1000),
        )
      } catch (transactionError) {
        console.error("Erro ao criar transação:", transactionError)

        // If error 400, mark as canceled
        const errorMessage = transactionError instanceof Error ? transactionError.message : String(transactionError)

        if (errorMessage.includes("400") || errorMessage.includes("Failed to create transaction")) {
          updateTradeInHistory(tradeId, {
            status: "Cancelada" as any,
            result: "Erro 400 - Transação cancelada",
          })
        } else {
          updateTradeInHistory(tradeId, {
            status: "Perdeu",
            result: "Erro na transação",
          })
        }

        setAutomation((prev) => ({ ...prev, waitingForResult: false, lastTransactionExpirationTime: null }))

        // Clear execution flags
        executingTradeRef.current = false
        setIsExecutingTrade(false)

        // Continue automation after error
        if (automationActiveRef.current) {
          console.log("🔄 Continuando automação após erro 400 em 5 segundos...")
          setTimeout(() => {
            if (automationActiveRef.current) {
              console.log("✅ Reiniciando automação após erro 400")
              executeAutomatedTrade(true, galeLevel, baseAmount)
            }
          }, 5000)
        }

        return // Exit function after error
      }
    } catch (error) {
      console.error("Erro no trade automatizado:", error)

      // Update existing trade with new opportunity status
      const currentAmount = calculateGaleAmount(baseAmount, galeMultiplier, galeLevel)
      const tradeId = addTradeToHistory({
        timestamp: new Date().toLocaleTimeString("pt-BR"),
        direction: "Buscando nova entrada...",
        amount: currentAmount.toString(), // Use the calculated Gale value
        status: "Nova oportunidade encontrada...",
        price: getCurrentPriceValue(),
      })

      setAutomation((prev) => ({ ...prev, waitingForResult: false, lastTransactionExpirationTime: null }))

      // Clear both execution flags
      console.log("🔓 Liberando flags de execução após erro geral")
      executingTradeRef.current = false
      setIsExecutingTrade(false)

      // Try to continue even after error if automation is still active
      if (automationActiveRef.current) {
        console.log("🔄 Continuando automação após erro geral em 10 segundos...")
        setTimeout(() => {
          if (automationActiveRef.current) {
            console.log("✅ Reiniciando automação após erro geral")

            // Reuse the same history entry for the new attempt
            updateTradeInHistory(tradeId, {
              status: "Analisando..." as any,
              direction: "",
              timestamp: new Date().toLocaleTimeString("pt-BR"),
            })

            // Execute new attempt reusing the same tradeId
            executeAutomatedTradeWithExistingId(true, galeLevel, baseAmount, tradeId)
          }
        }, 10000)
      }
    }
  }

  const executeAutomatedTradeWithExistingId = async (
    forceActive = false,
    galeLevel = currentGaleLevel,
    baseAmount = entryValue,
    existingTradeId: number,
  ) => {
    // Use BOTH: the ref and the state for double checking
    if (executingTradeRef.current || isExecutingTrade) {
      console.log("⚠️ BLOQUEADO: Trade já em execução, ignorando chamada duplicada")
      return
    }

    // Check if automation is active
    if (!forceActive && !automationActiveRef.current) {
      console.log("⚠️ Automação pausada, cancelando execução")
      return
    }

    // --- NEW LOGIC FOR INSUFFICIENT BALANCE CHECK ---
    const currentAmount = calculateGaleAmount(entryValue, galeMultiplier, galeLevel) // Value of the next entry
    const currentBalanceNum = Number.parseFloat(automation.currentBalance.replace(/\./g, "").replace(",", ".")) // Convert balance to number

    if (currentBalanceNum < currentAmount) {
      console.log(
        `🛑 Saldo insuficiente para a próxima operação. Saldo: R$ ${currentBalanceNum.toFixed(2)}, Necessário: R$ ${currentAmount.toFixed(2)}. Parando automação.`,
      )
      automationActiveRef.current = false // Stop automation loop
      setAutomation((prev) => ({ ...prev, isActive: false })) // Update state to inactive
      executingTradeRef.current = false
      setIsExecutingTrade(false)
      setShowInsufficientBalanceModal(true) // Show insufficient balance pop-up
      return
    }
    // --- END OF NEW LOGIC FOR INSUFFICIENT BALANCE CHECK ---

    // Set BOTH: ref and state to block other calls
    console.log("🔒 Definindo flags de execução para bloquear chamadas duplicadas")
    executingTradeRef.current = true
    setIsExecutingTrade(true)

    // --- NEW LOGIC FOR STOP LOSS CHECK BEFORE OPERATION ---
    const currentProfit = automation.currentProfit // Get current profit from state
    const stopLossValue = Number.parseFloat(stopLoss) // Convert stopLoss to number
    // currentAmount already calculated above

    const potentialProfitIfLoss = currentProfit - currentAmount // Calculate potential profit if this operation loses

    if (potentialProfitIfLoss <= -stopLossValue) {
      console.log(
        `🛑 Stop loss de R$ ${stopLossValue.toFixed(2)} seria atingido com a próxima operação (R$ ${currentAmount.toFixed(2)}). Lucro atual: R$ ${currentProfit.toFixed(2)}. Potencial: R$ ${potentialProfitIfLoss.toFixed(2)}. Parando automação.`,
      )
      automationActiveRef.current = false // Stop automation loop
      setAutomation((prev) => ({ ...prev, isActive: false })) // Update state to inactive

      // Add a special entry to history indicating Stop Loss was reached
      addTradeToHistory({
        timestamp: new Date().toLocaleTimeString("pt-BR"),
        direction: "STOP LOSS",
        amount: currentAmount.toFixed(2),
        status: "Cancelada", // Or a new status like "Stop Loss Reached"
        result: `Stop Loss Atingido: -R$ ${stopLossValue.toFixed(2)}`,
        price: getCurrentPriceValue(),
      })

      // Exit function, do not proceed with transaction
      executingTradeRef.current = false
      setIsExecutingTrade(false)
      return
    }
    // --- END OF NEW LOGIC FOR STOP LOSS CHECK ---

    console.log("🤖 Executando trade automatizado com ID existente...")

    try {
      // 1. CALCULATE GALE VALUE FIRST
      // currentAmount already calculated above
      console.log(`💰 DETALHES DO GALE ANTES DA OPERAÇÃO:`)
      console.log(`   - Gale Level atual: ${galeLevel}`)
      console.log(`   - Valor base: R$ ${baseAmount}`)
      console.log(`   - Multiplicador: ${galeMultiplier}`)
      console.log(`   - Valor calculado para esta operação: R$ ${currentAmount}`)

      // 2. Use the existing tradeId instead of creating a new one
      const tradeId = existingTradeId

      // Update with the correct Gale value
      updateTradeInHistory(tradeId, {
        amount: currentAmount.toString(),
        price: getCurrentPriceValue(),
        // No need to set transactionExpirationTime here, it's set after createTransaction
      })

      // ... rest of the code continues same as original executeAutomatedTrade ...
      // Simulate status progress
      setTimeout(() => {
        updateTradeInHistory(tradeId, { status: "Traçando estratégia..." })
      }, 1000)

      setTimeout(() => {
        updateTradeInHistory(tradeId, { status: "Executando..." })
      }, 2000)

      // 3. Get AI analysis
      const analysis = await getAIAnalysis()
      console.log("📊 Análise da IA:", analysis)

      // Update direction in history
      updateTradeInHistory(tradeId, {
        direction: analysis.recomendacao,
        status: "Aguardando resultado...",
      })

      // 4. Get current price
      const currentPrice = getCurrentPriceValue()

      // 5. Create transaction with already calculated Gale value
      try {
        const transaction = await createTransaction(analysis.recomendacao, currentPrice, currentAmount.toString())
        console.log("💰 Transação criada com valor:", currentAmount)

        // Update the trade history entry with the actual expiration time
        updateTradeInHistory(tradeId, {
          transactionExpirationTime: new Date(transaction.expiration_datetime).getTime(),
        })

        setAutomation((prev) => ({
          ...prev,
          lastTransactionId: transaction.transaction_id, // Use transaction.transaction_id
          waitingForResult: true,
          currentBalance: transaction.user_credit, // Use transaction.user_credit
          lastTransactionExpirationTime: new Date(transaction.expiration_datetime).getTime(), // Store as timestamp
        }))

        // Continue with the rest of the transaction logic...
        // 4. Wait for expiration + 2 seconds
        const expirationTime = new Date(transaction.expiration_datetime).getTime()
        const waitTime = expirationTime - Date.now() + 2000

        console.log(`⏰ Aguardando ${waitTime}ms para verificar resultado...`)

        setTimeout(
          async () => {
            try {
              // Check settlement
              const settlement = await checkSettlement()
              console.log("📈 Settlement:", settlement)

              // Update balance
              setAutomation((prev) => {
                // If the result starts with '+', it's positive profit
                // If the result starts with '-', it's a loss
                const isPositiveResult = settlement.amount_result.startsWith("+")
                const resultValue = Number.parseFloat(settlement.amount_result.replace(/[^\d.-]/g, ""))

                // If positive result, add the real value; if negative, subtract only the loss value
                const profitChange = isPositiveResult ? resultValue : -Number.parseFloat(entryValue)

                return {
                  ...prev,
                  currentBalance: settlement.user_credit,
                }
              })

              // 6. Wait 1 second and check transaction result
              setTimeout(async () => {
                try {
                  const result = await checkTransactionResult(transaction.transaction_id)

                  console.log("🎯 Resultado da transação:", result)

                  const won = result.transaction.status === "Ganhou"
                  const finalStatus = won ? "Ganhou" : "Perdeu"

                  // If won, play sound and show notification
                  if (won) {
                    playWinSound()

                    // Calculate won amount (returns - invested amount)
                    const returnsCents = result.transaction.returns_cents || 0
                    const amountCents = Number.parseFloat(currentAmount.toString()) * 100
                    const winAmount = ((returnsCents - amountCents) / 100).toFixed(2)

                    showWinNotification(`+R$ ${winAmount}`)
                  }

                  // Update history with final result
                  updateTradeInHistory(tradeId, {
                    status: finalStatus,
                    result: won
                      ? `+R$ ${((result.transaction.returns_cents - result.transaction.amount_cents) / 100).toFixed(2)}`
                      : settlement.amount_result,
                  })

                  // GALE LOGIC - APPLY FOR THE NEXT OPERATION
                  let nextGaleLevel = galeLevel
                  let nextEntryAmount = currentEntryAmount

                  if (won) {
                    // If won, reset Gale to base value
                    console.log("✅ Ganhou! Resetando Gale para valor base")
                    nextGaleLevel = 0
                    nextEntryAmount = entryValue
                    setCurrentGaleLevel(0)
                    setCurrentEntryAmount(entryValue)
                  } else {
                    // If lost, increase Gale level for the NEXT operation
                    nextGaleLevel = galeLevel + 1
                    const nextAmount = calculateGaleAmount(baseAmount, galeMultiplier, nextGaleLevel)
                    nextEntryAmount = nextAmount.toString()

                    console.log(`❌ Perdeu! Aumentando Gale de nível ${galeLevel} para nível ${nextGaleLevel}`)
                    console.log(
                      `🎯 Próxima entrada será: R$ ${nextAmount} (base: ${baseAmount}, multiplicador: ${galeMultiplier})`,
                    )

                    setCurrentGaleLevel(nextGaleLevel)
                    setCurrentEntryAmount(nextEntryAmount)
                  }

                  // Store values before updating state
                  let shouldContinue = false
                  let shouldStop = false

                  setAutomation((prev) => {
                    const newTotalTrades = prev.totalTrades + 1
                    const newWinTrades = won ? prev.winTrades + 1 : prev.winTrades
                    const newLossTrades = won ? prev.lossTrades : prev.lossTrades + 1
                    const newCurrentProfit = won
                      ? prev.currentProfit + (result.transaction.returns_cents - result.transaction.amount_cents) / 100
                      : prev.currentProfit - result.transaction.amount_cents / 100

                    // Update localStorage with the latest balance
                    const newBalance = result.transaction.user.balance
                    const storedUser = localStorage.getItem("user")
                    if (storedUser) {
                      try {
                        const parsedUser = JSON.parse(storedUser)
                        localStorage.setItem("user", JSON.stringify({ ...parsedUser, credit: newBalance }))
                      } catch (e) {
                        console.error("Error parsing stored user for balance update after trade:", e)
                      }
                    }

                    // Check if it should continue
                    const targetProfit = Number.parseFloat(profitTarget)
                    const stopLossValue = Number.parseFloat(stopLoss)

                    console.log(
                      `💰 Lucro atual: ${newCurrentProfit}, Meta: ${targetProfit}, Stop Loss: ${-stopLossValue}`,
                    )

                    // Check stop conditions
                    shouldStop = newCurrentProfit >= targetProfit || newCurrentProfit <= -stopLossValue

                    if (newCurrentProfit >= targetProfit) {
                      console.log("🎉 Meta de lucro atingida!")
                      showGoalNotification(newCurrentProfit.toFixed(2)) // Show goal notification
                      automationActiveRef.current = false
                    } else if (newCurrentProfit <= -stopLossValue) {
                      console.log("🛑 Stop loss atingido!")
                      automationActiveRef.current = false
                    } else {
                      // If stop conditions not met, should continue
                      shouldContinue = true
                      automationActiveRef.current = true
                      console.log(`🔄 Automação deve continuar: ${shouldContinue ? "SIM" : "NÃO"}`)
                    }

                    return {
                      ...prev,
                      totalTrades: newTotalTrades,
                      winTrades: newWinTrades,
                      lossTrades: newLossTrades,
                      currentProfit: newCurrentProfit,
                      waitingForResult: false,
                      currentBalance: result.transaction.user.balance,
                      isActive: shouldStop ? false : prev.isActive,
                      lastTransactionExpirationTime: null, // Clear expiration time after result
                    }
                  })

                  // IMPORTANT: Clear execution flags BEFORE scheduling the next execution
                  executingTradeRef.current = false
                  setIsExecutingTrade(false)

                  // Explicit check for debugging
                  console.log(
                    `🔍 Estado da automação: automationActiveRef=${automationActiveRef.current}, shouldStop=${shouldStop}, shouldContinue=${shouldContinue}`,
                  )
                  console.log(`🎲 Próximo Gale Level: ${nextGaleLevel}, Próximo valor: R$ ${nextEntryAmount}`)

                  // Force continuation if stop conditions not met
                  const shouldForceContinue = !shouldStop && automationActiveRef.current

                  // Continue only if the flag is set and there are no stop conditions
                  if (shouldForceContinue) {
                    console.log("🔄 Continuando automação em 5 segundos...")
                    // Use setTimeout to give a longer interval between executions
                    setTimeout(() => {
                      if (automationActiveRef.current) {
                        console.log(`✅ Iniciando próximo trade automatizado com Gale Level ${nextGaleLevel}`)
                        // Pass the correct Gale values for the next execution
                        executeAutomatedTrade(true, nextGaleLevel, baseAmount)
                      } else {
                        console.log("❌ Automação foi pausada durante o intervalo, cancelando próximo trade")
                      }
                    }, 5000)
                  } else {
                    console.log("⏹️ Automação finalizada ou pausada, não agendando próximo trade")
                  }
                } catch (error) {
                  console.error("Erro ao verificar resultado:", error)
                  updateTradeInHistory(tradeId, { status: "Perdeu", result: "Erro" })
                  setAutomation((prev) => ({ ...prev, waitingForResult: false, lastTransactionExpirationTime: null }))
                  // Clear both execution flags
                  executingTradeRef.current = false
                  setIsExecutingTrade(false)
                }
              }, 1000)
            } catch (error) {
              console.error("Erro ao verificar settlement:", error)
              updateTradeInHistory(tradeId, { status: "Perdeu", result: "Erro" })
              setAutomation((prev) => ({ ...prev, waitingForResult: false, lastTransactionExpirationTime: null }))
              // Clear both execution flags
              console.log("🔓 Liberando flags de execução após erro")
              executingTradeRef.current = false
              setIsExecutingTrade(false)

              // Try to reconnect after error if automation is still active
              if (automationActiveRef.current) {
                console.log("🔄 Continuando automação após erro em 10 segundos...")
                setTimeout(() => {
                  if (automationActiveRef.current) {
                    console.log("✅ Reiniciando automação após erro")
                    executeAutomatedTrade()
                  }
                }, 10000)
              }
            }
          },
          Math.max(waitTime, 1000),
        )
      } catch (transactionError) {
        console.error("Erro ao criar transação:", transactionError)

        // If error 400, mark as canceled
        const errorMessage = transactionError instanceof Error ? transactionError.message : String(transactionError)

        if (errorMessage.includes("400") || errorMessage.includes("Failed to create transaction")) {
          updateTradeInHistory(tradeId, {
            status: "Cancelada" as any,
            result: "Erro 400 - Transação cancelada",
          })
        } else {
          updateTradeInHistory(tradeId, {
            status: "Perdeu",
            result: "Erro na transação",
          })
        }

        setAutomation((prev) => ({ ...prev, waitingForResult: false, lastTransactionExpirationTime: null }))

        // Clear execution flags
        executingTradeRef.current = false
        setIsExecutingTrade(false)

        // Continue automation after error
        if (automationActiveRef.current) {
          console.log("🔄 Continuando automação após erro 400 em 5 segundos...")
          setTimeout(() => {
            if (automationActiveRef.current) {
              console.log("✅ Reiniciando automação após erro 400")
              executeAutomatedTrade(true, galeLevel, baseAmount)
            }
          }, 5000)
        }

        return // Exit function after error
      }
    } catch (error) {
      console.error("Erro no trade automatizado:", error)

      // Update existing trade with new opportunity status
      const currentAmount = calculateGaleAmount(baseAmount, galeMultiplier, galeLevel)
      const tradeId = addTradeToHistory({
        timestamp: new Date().toLocaleTimeString("pt-BR"),
        direction: "Buscando nova entrada...",
        amount: currentAmount.toString(), // Use the calculated Gale value
        status: "Nova oportunidade encontrada...",
        price: getCurrentPriceValue(),
      })

      setAutomation((prev) => ({ ...prev, waitingForResult: false, lastTransactionExpirationTime: null }))

      // Clear both execution flags
      console.log("🔓 Liberando flags de execução após erro geral")
      executingTradeRef.current = false
      setIsExecutingTrade(false)

      // Try to continue even after error if automation is still active
      if (automationActiveRef.current) {
        console.log("🔄 Continuando automação após erro geral em 10 segundos...")
        setTimeout(() => {
          if (automationActiveRef.current) {
            console.log("✅ Reiniciando automação após erro geral")

            // Reuse the same history entry for the new attempt
            updateTradeInHistory(tradeId, {
              status: "Analisando..." as any,
              direction: "",
              timestamp: new Date().toLocaleTimeString("pt-BR"),
            })

            // Execute new attempt reusing the same tradeId
            executeAutomatedTradeWithExistingId(true, galeLevel, baseAmount, tradeId)
          }
        }, 10000)
      }
    }
  }

  const toggleAutomation = () => {
    if (automation.isActive) {
      // Pause automation immediately
      console.log("⏸️ Pausando automação...")
      automationActiveRef.current = false
      setAutomation((prev) => ({ ...prev, isActive: false }))
      // Ensure both execution flags are disabled
      executingTradeRef.current = false
      setIsExecutingTrade(false)
    } else {
      // Start automation - clear history
      setTradeHistory([])

      // Check if there's already a trade in progress
      if (executingTradeRef.current || isExecutingTrade) {
        console.log("⚠️ Já existe um trade em execução, aguardando finalização antes de iniciar automação")
        return
      }

      // Update the ref before changing states to avoid race conditions
      automationActiveRef.current = true

      // Update automation state
      setAutomation((prev) => ({
        ...prev,
        isActive: true,
        currentProfit: 0,
        totalTrades: 0,
        winTrades: 0,
        lossTrades: 0,
        lastTransactionExpirationTime: null, // Reset expiration time
      }))

      // Reset Gale when starting new automation
      setCurrentGaleLevel(0)
      setCurrentEntryAmount(entryValue)

      // Ensure execution flags are disabled before starting
      executingTradeRef.current = false
      setIsExecutingTrade(false)

      // Use setTimeout to avoid multiple simultaneous executions
      console.log("🚀 Agendando primeiro trade automatizado em 500ms...")
      setTimeout(() => {
        // Check again if automation is still active
        if (automationActiveRef.current) {
          console.log("▶️ Executando primeiro trade automatizado")
          executeAutomatedTrade(true) // Force execution the first time
        } else {
          console.log("❌ Automação foi desativada durante o intervalo")
        }
      }, 500)
    }
  }

  const drawChart = () => {
    const canvas = canvasRef.current
    if (!canvas || chartData.length === 0) return

    const ctx = canvas.getContext("2d")
    if (!ctx) return

    // Set canvas size to fill the container completely
    const container = containerRef.current
    if (container) {
      const containerRect = container.getBoundingClientRect()
      canvas.width = containerRect.width
      canvas.height = Math.max(250, Math.min(400, containerRect.width * 0.5))

      // Set CSS size to match canvas size for proper scaling
      canvas.style.width = `${containerRect.width}px`
      canvas.style.height = `${canvas.height}px`
    }

    const { width, height } = canvas
    const padding = Math.max(50, width * 0.08)
    const bottomPadding = Math.max(70, height * 0.15)

    // Clear canvas
    ctx.fillStyle = "#1f2937"
    ctx.fillRect(0, 0, width, height)

    // Calculate price range with padding
    const prices = chartData.flatMap((d) => [
      Number.parseFloat(d.open),
      Number.parseFloat(d.higher),
      Number.parseFloat(d.lower),
      Number.parseFloat(d.close),
    ])
    const minPrice = Math.min(...prices)
    const maxPrice = Math.max(...prices)
    const priceRange = maxPrice - minPrice
    const paddedMin = minPrice - priceRange * 0.05
    const paddedMax = maxPrice + priceRange * 0.05
    const paddedRange = paddedMax - paddedMin

    // Draw grid lines
    ctx.strokeStyle = "#374151"
    ctx.lineWidth = 1

    // Horizontal grid lines
    for (let i = 0; i <= 8; i++) {
      const y = padding + (height - padding - bottomPadding) * (i / 8)
      ctx.beginPath()
      ctx.moveTo(padding, y)
      ctx.lineTo(width - padding, y)
      ctx.stroke()
    }

    // Vertical grid lines
    const verticalLines = Math.max(6, Math.floor(chartData.length / 3))
    for (let i = 0; i <= verticalLines; i++) {
      const x = padding + (width - 2 * padding) * (i / verticalLines)
      ctx.beginPath()
      ctx.moveTo(x, padding)
      ctx.lineTo(x, height - bottomPadding)
      ctx.stroke()
    }

    // Draw candlesticks
    const candleWidth = Math.max(3, ((width - 2 * padding) / chartData.length) * 0.7)
    const closePoints: { x: number; y: number }[] = []

    chartData.forEach((candle, index) => {
      const x = padding + (width - 2 * padding) * (index / (chartData.length - 1))

      const open = Number.parseFloat(candle.open)
      const close = Number.parseFloat(candle.close)
      const high = Number.parseFloat(candle.higher)
      const low = Number.parseFloat(candle.lower)

      const openY = padding + ((paddedMax - open) / paddedRange) * (height - padding - bottomPadding)
      const closeY = padding + ((paddedMax - close) / paddedRange) * (height - padding - bottomPadding)
      const highY = padding + ((paddedMax - high) / paddedRange) * (height - padding - bottomPadding)
      const lowY = padding + ((paddedMax - low) / paddedRange) * (height - padding - bottomPadding)

      // Store close points for trend line
      closePoints.push({ x, y: closeY })

      // Determine candle color
      const isGreen = close > open
      ctx.fillStyle = isGreen ? "#10b981" : "#ef4444"
      ctx.strokeStyle = isGreen ? "#10b981" : "#ef4444"

      // Draw wick
      ctx.lineWidth = 2
      ctx.beginPath()
      ctx.moveTo(x, highY)
      ctx.lineTo(x, lowY)
      ctx.stroke()

      // Draw body
      const bodyTop = Math.min(openY, closeY)
      const bodyHeight = Math.abs(closeY - openY)
      ctx.fillRect(x - candleWidth / 2, bodyTop, candleWidth, Math.max(bodyHeight, 2))
    })

    // Draw trend line connecting close prices
    if (closePoints.length > 1) {
      ctx.strokeStyle = "#3b82f6"
      ctx.lineWidth = 2
      ctx.setLineDash([5, 5])
      ctx.beginPath()
      ctx.moveTo(closePoints[0].x, closePoints[0].y)
      for (let i = 1; i < closePoints.length; i++) {
        ctx.lineTo(closePoints[i].x, closePoints[i].y)
      }
      ctx.stroke()
      ctx.setLineDash([])
    }

    // Draw price labels with better visibility
    ctx.fillStyle = "#ffffff"
    ctx.font = `${Math.max(9, Math.min(12, width * 0.012))}px Arial`
    ctx.textAlign = "right"
    ctx.strokeStyle = "#1f2937"
    ctx.lineWidth = 2

    for (let i = 0; i <= 8; i++) {
      const price = paddedMax - (paddedRange * i) / 8
      const y = padding + (height - padding - bottomPadding) * (i / 8)
      const priceText = price.toString()

      // Draw text outline for better visibility
      ctx.strokeText(priceText, padding - 20, y + 4)
      ctx.fillText(priceText, padding - 20, y + 4)
    }

    // Draw time labels with better visibility
    ctx.textAlign = "center"
    ctx.fillStyle = "#ffffff"
    ctx.font = `${Math.max(8, Math.min(11, width * 0.011))}px Arial`

    // Draw time labels for each candle position (moving with the chart)
    chartData.forEach((candle, index) => {
      const x = padding + (width - 2 * padding) * (index / (chartData.length - 1))

      // Only show time labels for every few candles to avoid overcrowding
      if (index % Math.max(1, Math.floor(chartData.length / 5)) === 0 || index === chartData.length - 1) {
        const time = formatTime(candle.open_time)

        // Draw text outline for better visibility
        ctx.strokeText(time, x, height - 30)
        ctx.fillText(time, x, height - 30)
      }
    })

    // Draw current price indicator
    if (chartData.length > 0) {
      const currentPrice = Number.parseFloat(chartData[chartData.length - 1].close)
      const currentY = padding + ((paddedMax - currentPrice) / paddedRange) * (height - padding - bottomPadding)

      // Draw horizontal line for current price
      ctx.strokeStyle = "#fbbf24"
      ctx.lineWidth = 2
      ctx.setLineDash([10, 5])
      ctx.beginPath()
      ctx.moveTo(padding, currentY)
      ctx.lineTo(width - padding, currentY)
      ctx.stroke()
      ctx.setLineDash([])

      // Draw current price label
      ctx.fillStyle = "#fbbf24"
      ctx.font = `${Math.max(9, Math.min(12, width * 0.012))}px Arial`
      ctx.textAlign = "left"
      ctx.strokeStyle = "#1f2937"
      ctx.lineWidth = 2
      const currentPriceText = currentPrice.toString()
      ctx.strokeText(currentPriceText, width - padding + 15, currentY + 4)
      ctx.fillText(currentPriceText, width - padding + 15, currentY + 4)
    }
  }

  const handleMouseMove = (event: React.MouseEvent<HTMLCanvasElement>) => {
    const canvas = canvasRef.current
    if (!canvas || chartData.length === 0) return

    const rect = canvas.getBoundingClientRect()
    const x = event.clientX - rect.left
    const y = event.clientY - rect.top

    const { width, height } = canvas
    const padding = width < 768 ? 30 : 40

    // Find the closest candle
    const candleIndex = Math.round(((x - padding) / (width - 2 * padding)) * (chartData.length - 1))

    if (candleIndex >= 0 && candleIndex < chartData.length) {
      const candle = chartData[candleIndex]
      setTooltip({
        x: event.clientX,
        y: event.clientY,
        candle,
        visible: true,
      })
    }
  }

  const handleMouseLeave = () => {
    setTooltip((prev) => ({ ...prev, visible: false }))
  }

  return (
    <PageTransition>
      <div className="min-h-screen bg-gradient-to-br from-gray-950 via-slate-900 to-gray-950">
        {/* Header */}
        <header className="border-b border-gray-800/50 bg-gray-900/30 backdrop-blur-xl">
          <div className="container mx-auto px-4 py-4">
            <div className="flex items-center justify-between">
              <div className="flex items-center space-x-4">
                <Button
                  variant="ghost"
                  size="icon"
                  className="text-gray-400 hover:text-white hover:bg-gray-800"
                  onClick={() => handleNavigationAttempt("/ai-operation")}
                >
                  <ArrowLeft className="w-5 h-5" />
                </Button>
                <div className="flex items-center space-x-3">
                  <div className="p-2 bg-gray-800 rounded-lg">
                    <Image
                      src="/logo-optimus.png"
                      alt="OptimusIA Logo"
                      width={32}
                      height={32}
                      className="drop-shadow-lg"
                    />
                  </div>
                  <div>
                    <span className="text-xl font-bold text-white">OptimusIA</span>
                    <div className="flex items-center space-x-2 mt-1">
                      <Zap className="w-4 h-4 text-cyan-400" />
                      <span className="text-sm text-cyan-400 font-medium">AUTOMAÇÃO AVANÇADA</span>
                    </div>
                  </div>
                </div>
              </div>
              {/* Refresh Button with cooldown */}
              <Button
                variant="ghost"
                size="icon"
                className="text-gray-400 hover:text-white hover:bg-gray-800"
                onClick={fetchUserBalance}
                disabled={refreshCooldown > 0}
              >
                <RefreshCw className="w-5 h-5" />
                {refreshCooldown > 0 && <span className="ml-1 text-xs">{refreshCooldown}s</span>}
              </Button>
            </div>
          </div>
        </header>

        {/* Main Content */}
        <main className="container mx-auto px-4 py-6">
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
            {/* Chart Section */}
            <div className="lg:col-span-2 space-y-6">
              {/* Asset Info */}
              <Card className="bg-gray-900/80 border-gray-800 backdrop-blur-xl shadow-xl">
                <CardContent className="p-4">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center space-x-3">
                      <div className="flex items-center justify-center w-12 h-12 rounded-xl bg-gray-800 overflow-hidden">
                        {imageUrl ? (
                          <img src={imageUrl || "/placeholder.svg"} alt={name || ""} className="w-8 h-8" />
                        ) : (
                          <span className="text-lg font-bold text-white">{symbol?.substring(0, 2)}</span>
                        )}
                      </div>
                      <div>
                        <h2 className="text-xl font-bold text-white">{name}</h2>
                        <p className="text-gray-400 text-sm">{symbol}</p>
                      </div>
                    </div>
                    <div className="text-right">
                      <p className="text-2xl font-bold text-white">${getCurrentPriceValue()}</p>
                      <div className={`flex items-center space-x-1 ${isPositive ? "text-green-400" : "text-red-400"}`}>
                        {isPositive ? <TrendingUp className="w-4 h-4" /> : <TrendingDown className="w-4 h-4" />}
                        <span className="font-semibold text-sm">
                          {isPositive ? "+" : ""}
                          {change.toFixed(2)} ({isPositive ? "+" : ""}
                          {percentage.toFixed(2)}%)
                        </span>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>

              {/* Chart */}
              <Card className="bg-gray-900/80 border-gray-800 backdrop-blur-xl shadow-xl">
                <CardContent className="p-4">
                  <div className="flex flex-col md:flex-row md:items-center md:justify-between mb-4 space-y-3 md:space-y-0">
                    <div className="grid grid-cols-4 gap-2 md:gap-4 text-xs md:text-sm bg-gray-800/50 p-2 rounded-lg">
                      <div className="text-center">
                        <p className="text-gray-300 font-medium text-xs mb-1">OPEN</p>
                        <p className="text-white font-semibold text-sm md:text-base bg-gray-700 px-1 py-1 rounded">
                          {ohlc.open}
                        </p>
                      </div>
                      <div className="text-center">
                        <p className="text-gray-300 font-medium text-xs mb-1">HIGH</p>
                        <p className="text-green-400 font-semibold text-sm md:text-base bg-gray-700 px-1 py-1 rounded">
                          {ohlc.high}
                        </p>
                      </div>
                      <div className="text-center">
                        <p className="text-gray-300 font-medium text-xs mb-1">LOW</p>
                        <p className="text-red-400 font-semibold text-sm md:text-base bg-gray-700 px-1 py-1 rounded">
                          {ohlc.low}
                        </p>
                      </div>
                      <div className="text-center">
                        <p className="text-gray-300 font-medium text-xs mb-1">CLOSE</p>
                        <p className="text-white font-semibold text-sm md:text-base bg-gray-700 px-1 py-1 rounded">
                          {ohlc.close}
                        </p>
                      </div>
                    </div>
                    {/* Status do WebSocket */}
                    <div
                      className={`flex items-center space-x-2 ${wsStatus === "connected" ? "bg-green-500/20" : wsStatus === "connecting" ? "bg-yellow-500/20" : "bg-red-500/20"} px-2 py-1 rounded-lg`}
                    >
                      <div
                        className={`w-2 h-2 rounded-full ${getWSStatusColor()} ${wsStatus === "connected" ? "animate-pulse" : ""}`}
                      ></div>
                      <span
                        className={`text-xs md:text-sm font-medium ${wsStatus === "connected" ? "text-green-400" : wsStatus === "connecting" ? "text-yellow-400" : "text-red-400"}`}
                      >
                        {getWSStatusText()}
                      </span>
                    </div>
                  </div>

                  {isLoading ? (
                    <div className="flex items-center justify-center h-64 md:h-96">
                      <div className="text-center">
                        <div className="animate-spin rounded-full h-8 w-8 md:h-12 md:w-12 border-b-2 border-blue-500 mx-auto mb-4"></div>
                        <p className="text-gray-400 text-sm md:text-base">Carregando gráfico...</p>
                      </div>
                    </div>
                  ) : error ? (
                    <div className="flex items-center justify-center h-64 md:h-96">
                      <div className="text-center">
                        <p className="text-red-400 mb-4 text-sm md:text-base">{error}</p>
                        <Button
                          variant="outline"
                          className="border-gray-700 text-gray-300 hover:bg-gray-800 text-sm md:text-base bg-transparent"
                          onClick={() => symbol && fetchInitialChartData(symbol)}
                        >
                          Tentar Novamente
                        </Button>
                      </div>
                    </div>
                  ) : (
                    <div className="relative w-full" ref={containerRef}>
                      <canvas
                        ref={canvasRef}
                        className="w-full h-auto rounded-lg cursor-crosshair block"
                        style={{ background: "#1f2937" }}
                        onMouseMove={handleMouseMove}
                        onMouseLeave={handleMouseLeave}
                      />

                      {/* Tooltip */}
                      {tooltip.visible && (
                        <div
                          className="absolute z-10 bg-gray-800 border border-gray-700 rounded-md shadow-lg p-2 text-sm text-gray-300"
                          style={{
                            left: tooltip.x + 10,
                            top: tooltip.y + 10,
                          }}
                        >
                          <p>
                            Abertura: <b>{tooltip.candle.open}</b>
                          </p>
                          <p>
                            Fechamento: <b>{tooltip.candle.close}</b>
                          </p>
                          <p>
                            Máxima: <b>{tooltip.candle.higher}</b>
                          </p>
                          <p>
                            Mínima: <b>{tooltip.candle.lower}</b>
                          </p>
                          <p>
                            Volume: <b>{tooltip.candle.volume}</b>
                          </p>
                          <p>
                            Horário: <b>{formatTime(tooltip.candle.open_time)}</b>
                          </p>
                        </div>
                      )}
                    </div>
                  )}
                </CardContent>
              </Card>
            </div>

            {/* Automation Settings */}
            <div className="lg:col-span-1 space-y-6">
              <Card className="bg-gray-900/80 border-gray-800 backdrop-blur-xl shadow-xl">
                <CardContent className="p-4 space-y-4">
                  <h3 className="text-lg font-semibold text-white">Configurações</h3>

                  {/* Entry Value */}
                  <div>
                    <Label htmlFor="entryValue" className="text-gray-300">
                      Valor da Entrada
                    </Label>
                    <Input
                      type="number"
                      id="entryValue"
                      className="bg-gray-800 border-gray-700 text-white"
                      value={entryValue}
                      onChange={(e) => setEntryValue(e.target.value.replace(",", "."))}
                    />
                  </div>

                  {/* Gale Multiplier */}
                  <div>
                    <Label htmlFor="galeMultiplier" className="text-gray-300">
                      Multiplicador de Gale
                    </Label>
                    <Select value={galeMultiplier} onValueChange={setGaleMultiplier}>
                      <SelectTrigger className="bg-gray-800 border-gray-700 text-white">
                        <SelectValue placeholder="Selecione" />
                      </SelectTrigger>
                      <SelectContent className="bg-gray-800 border-gray-700 text-white">
                        <SelectItem value="1.1">1.1 (Conservador)</SelectItem>
                        <SelectItem value="1.2">1.2 (Moderado)</SelectItem>
                        <SelectItem value="1.3">1.3 (Agressivo)</SelectItem>
                        <SelectItem value="1.4">1.4 (Muito Agressivo)</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>

                  {/* Expiration */}
                  <div>
                    <Label htmlFor="expiration" className="text-gray-300">
                      Expiração (minutos)
                    </Label>
                    <Select value={expiration} onValueChange={setExpiration}>
                      <SelectTrigger className="bg-gray-800 border-gray-700 text-white">
                        <SelectValue placeholder="Selecione" />
                      </SelectTrigger>
                      <SelectContent className="bg-gray-800 border-gray-700 text-white">
                        <SelectItem value="1">1 minuto</SelectItem>
                        <SelectItem value="5">5 minutos</SelectItem>
                        <SelectItem value="10">10 minutos</SelectItem>
                        <SelectItem value="15">15 minutos</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>

                  {/* Profit Target */}
                  <div>
                    <Label htmlFor="profitTarget" className="text-gray-300">
                      Meta de Lucro (R$)
                    </Label>
                    <Input
                      type="number"
                      id="profitTarget"
                      className="bg-gray-800 border-gray-700 text-white"
                      value={profitTarget}
                      onChange={(e) => setProfitTarget(e.target.value.replace(",", "."))}
                    />
                  </div>

                  {/* Stop Loss */}
                  <div>
                    <Label htmlFor="stopLoss" className="text-gray-300">
                      Stop Loss (R$)
                    </Label>
                    <Input
                      type="number"
                      id="stopLoss"
                      className="bg-gray-800 border-gray-700 text-white"
                      value={stopLoss}
                      onChange={(e) => setStopLoss(e.target.value.replace(",", "."))}
                    />
                  </div>

                  {/* Toggle Automation Button */}
                  <Button
                    className="w-full bg-blue-500 hover:bg-blue-600 text-white font-semibold"
                    onClick={toggleAutomation}
                    disabled={automation.waitingForResult || wsStatus !== "connected"}
                  >
                    {automation.isActive ? (
                      <>
                        <Pause className="w-4 h-4 mr-2" />
                        Pausar Automação
                      </>
                    ) : (
                      <>
                        <Play className="w-4 h-4 mr-2" />
                        Iniciar Automação
                      </>
                    )}
                  </Button>
                </CardContent>
              </Card>

              {/* Balance Info */}
              <Card className="bg-gray-900/80 border-gray-800 backdrop-blur-xl shadow-xl">
                <CardContent className="p-4 space-y-4">
                  <h3 className="text-lg font-semibold text-white">Saldo</h3>
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-sm text-gray-400">Saldo Atual</p>
                      <p className="text-2xl font-bold text-white">R$ {automation.currentBalance}</p>
                    </div>
                    <DollarSign className="w-8 h-8 text-green-400" />
                  </div>
                </CardContent>
              </Card>
            </div>
          </div>

          {/* Total Profit Info */}
          <Card className="bg-gray-900/80 border-gray-800 backdrop-blur-xl shadow-xl mb-6">
            <CardContent className="p-4 space-y-4">
              <h3 className="text-lg font-semibold text-white">Lucro Total</h3>
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-gray-400">Lucro/Prejuízo Atual</p>
                  <p
                    className={`text-2xl font-bold ${automation.currentProfit >= 0 ? "text-green-400" : "text-red-400"}`}
                  >
                    {automation.currentProfit >= 0 ? "+" : ""}R$ {automation.currentProfit.toFixed(2)}
                  </p>
                </div>
                <Target className="w-8 h-8 text-purple-400" />
              </div>
            </CardContent>
          </Card>

          {/* Trade History */}
          <Card className="bg-gray-900/80 border-gray-800 backdrop-blur-xl shadow-xl">
            <CardContent className="p-6">
              <h3 className="text-lg font-semibold text-white mb-4">Histórico de Operações</h3>
              <div className="space-y-3 max-h-64 overflow-y-auto">
                {tradeHistory.length === 0 ? (
                  <p className="text-gray-400 text-center py-4">Nenhuma operação realizada ainda</p>
                ) : (
                  tradeHistory.map((trade) => (
                    <div key={trade.id} className="flex items-center justify-between p-3 bg-gray-800/50 rounded-lg">
                      <div className="flex items-center space-x-3">
                        {getStatusIcon(trade.status)}
                        <div>
                          <div className="flex items-center space-x-2">
                            <span className="text-white font-medium">{trade.direction || "..."}</span>
                            <span className="text-gray-400">•</span>
                            <span className="text-gray-400">R$ {trade.amount}</span>
                            {/* Display countdown if status is executing/waiting and time is available */}
                            {(trade.status === "Executando..." || trade.status === "Aguardando resultado...") &&
                              remainingTimes[trade.id] !== undefined && (
                                <>
                                  <span className="text-gray-400 hidden sm:inline">•</span>
                                  <span className="text-yellow-400 font-semibold text-sm md:text-base">
                                    Expiração:{" "}
                                    {Math.floor(remainingTimes[trade.id] / 60)
                                      .toString()
                                      .padStart(2, "0")}
                                    :{(remainingTimes[trade.id] % 60).toString().padStart(2, "0")}
                                  </span>
                                </>
                              )}
                          </div>
                          <p className="text-xs text-gray-500">{trade.timestamp}</p>
                        </div>
                      </div>
                      <div className="text-right">
                        <p className={`font-semibold ${getStatusColor(trade.status)}`}>{trade.status}</p>
                        {trade.result && (
                          <p className={`text-xs ${trade.result.startsWith("+") ? "text-green-400" : "text-red-400"}`}>
                            {trade.result}
                          </p>
                        )}
                      </div>
                    </div>
                  ))
                )}
              </div>
            </CardContent>
          </Card>
        </main>
      </div>
      {/* Win Notification */}
      {winNotification && (
        <div
          className={`fixed bottom-6 left-4 md:left-6 z-50 transform transition-all duration-300 ease-out ${
            winNotification.visible ? "translate-x-0 opacity-100 scale-100" : "-translate-x-full opacity-0 scale-95"
          }`}
        >
          <div className="bg-gradient-to-r from-green-500 to-emerald-500 text-white px-4 py-3 md:px-6 md:py-4 rounded-xl shadow-2xl border border-green-400/30 backdrop-blur-sm max-w-xs md:max-w-sm">
            <div className="flex items-center space-x-2 md:space-x-3">
              <div className="w-6 h-6 md:w-8 md:h-8 bg-white/20 rounded-full flex items-center justify-center flex-shrink-0">
                <CheckCircle className="w-4 h-4 md:w-5 md:h-5 text-white" />
              </div>
              <div className="min-w-0 flex-1">
                <p className="font-bold text-base md:text-lg truncate">{winNotification.amount}</p>
                <p className="text-green-100 text-xs md:text-sm">Operação vencedora!</p>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Goal Notification */}
      {goalNotification && (
        <div
          className={`fixed bottom-6 left-4 md:left-6 z-50 transform transition-all duration-300 ease-out ${
            goalNotification.visible ? "translate-x-0 opacity-100 scale-100" : "-translate-x-full opacity-0 scale-95"
          }`}
        >
          <div className="bg-gradient-to-r from-purple-500 to-indigo-500 text-white px-4 py-3 md:px-6 md:py-4 rounded-xl shadow-2xl border border-purple-400/30 backdrop-blur-sm max-w-xs md:max-w-sm flex flex-col items-start">
            <div className="flex items-center space-x-2 md:space-x-3 mb-2">
              <div className="w-6 h-6 md:w-8 md:h-8 bg-white/20 rounded-full flex items-center justify-center flex-shrink-0">
                <Target className="w-4 h-4 md:w-5 md:h-5 text-white" />
              </div>
              <div className="min-w-0 flex-1">
                <p className="font-bold text-base md:text-lg truncate">Meta Atingida: R$ {goalNotification.amount}</p>
                <p className="text-purple-100 text-xs md:text-sm">Parabéns! Sua meta de lucro foi alcançada.</p>
              </div>
            </div>
            <Button
              onClick={dismissGoalNotification}
              className="w-full bg-purple-700 hover:bg-purple-800 text-white text-sm py-1.5 mt-2"
            >
              Entendi
            </Button>
          </div>
        </div>
      )}

      {/* Leave Confirmation AlertDialog */}
      <AlertDialog open={showConfirmLeaveModal} onOpenChange={setShowConfirmLeaveModal}>
        <AlertDialogContent className="bg-gray-900 border-gray-800 text-white">
          <AlertDialogHeader>
            <AlertDialogTitle className="text-white">Operação em Andamento</AlertDialogTitle>
            <AlertDialogDescription className="text-gray-400">
              Uma operação de automação está ativa e aguardando o resultado. Sair agora pode interromper a operação.
              {automation.lastTransactionExpirationTime && remainingTimeForTrade > 0 && (
                <span className="mt-2 text-lg font-bold text-blue-400 block">
                  Tempo restante: {remainingTimeForTrade} segundos
                </span>
              )}
              {automation.lastTransactionExpirationTime && remainingTimeForTrade <= 0 && (
                <span className="mt-2 text-lg font-bold text-yellow-400 block">Aguardando resultado final...</span>
              )}
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel
              className="bg-gray-700 hover:bg-gray-600 text-white border-gray-600"
              onClick={() => {
                setShowConfirmLeaveModal(false)
                setNextNavigationPath(null)
              }}
            >
              Permanecer
            </AlertDialogCancel>
            <AlertDialogAction
              className="bg-red-600 hover:bg-red-700 text-white"
              onClick={() => {
                // Stop automation if user decides to leave
                automationActiveRef.current = false
                setAutomation((prev) => ({
                  ...prev,
                  isActive: false,
                  waitingForResult: false,
                  lastTransactionExpirationTime: null,
                }))
                executingTradeRef.current = false
                setIsExecutingTrade(false)

                // Navigate to the intended path
                if (nextNavigationPath) {
                  router.push(nextNavigationPath)
                }
                setShowConfirmLeaveModal(false)
                setNextNavigationPath(null)
              }}
            >
              Sair Mesmo Assim
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>

      {/* Insufficient Balance AlertDialog */}
      <AlertDialog open={showInsufficientBalanceModal} onOpenChange={setShowInsufficientBalanceModal}>
        <AlertDialogContent className="bg-gray-900 border-gray-800 text-white">
          <AlertDialogHeader>
            <AlertDialogTitle className="text-white">Saldo Insuficiente</AlertDialogTitle>
            <AlertDialogDescription className="text-gray-400">
              Não há saldo suficiente para efetuar a próxima entrada. A automação foi pausada.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogAction
              className="bg-blue-600 hover:bg-blue-700 text-white"
              onClick={() => setShowInsufficientBalanceModal(false)}
            >
              Entendi
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </PageTransition>
  )
}
